/*
Copyright (C) 2011 Simon Lehmayr <simon.lehmayr@gmx.de>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/** standard error fn. if cond is true an error has occurred, so the text is printed ans program aborts
    @param cond - {[false, true]} true means error
    @param text - error message
*/
static void error(int cond, const char * text)
{
    if (cond) {
        printf("%s\n", text);
        exit(-1);
    }
}

/** general wave chunk eater. parses chunks in wave files.
    @param f - file to parse
    @param s - string name of current chunk
    @param readsize {[false, true]} - true means chunk size is to be read (and skipped)
    @param jump {[false, true]} - true means that chunk will be skipped
    @return start position of current chunk data (after chunk header)
*/
static unsigned long eat_chunk(FILE * f, char *s, int readsize, int jump)
{
    unsigned long size, pos;
    fread(s, 1, 4, f);
    if (readsize) fread(&size, 1, 4, f);
    pos = ftell(f);
    if (readsize && jump) fseek(f, size, SEEK_CUR);
    return pos;
}

/** header skipper. returns the start position of raw wave sample data.
    @param f - file to process
    @return start position of raw data
*/
static unsigned long determine_offset(FILE *f, char *name)
{
    char s[5];
    unsigned long pos;
    fseek(f, 0, SEEK_SET);
    eat_chunk(f, s, 1, 0);
    error(strncmp(s, "RIFF", 4) != 0, "not a wave file!");
    eat_chunk(f, s, 0, 0);
    error(strncmp(s, "WAVE", 4) != 0, "invalid wave file!");
    while(1) {
        pos = eat_chunk(f, s, 1, 1);
        error(feof(f), "invalid wave file!");
        if (strncmp(s, name, 4) == 0) break; 
    }
    return pos;
}


#define WAVEXSIZE 0x3c

struct fmt_st {
    unsigned short wFormatTag;
    unsigned short wChannels;
    unsigned long dwSamplesPerSec;
    unsigned long dwAvgBytesPerSec;
    unsigned short wBlockAlign;
    unsigned short wBitsPerSample;
};

void main(int argc, char *args[])
{
    FILE *ff, *fr, *fo;
    struct fmt_st fmt_s;
    unsigned long sizeff, sizefr, fosize, s, samplerate, samplewidth, avgbytessec, blockalign, bitpersample, offsf, offsr, hpos;
    double realp, curp;
    char buf[18], foname[2048], frname[2048];
    char hedr[]={'R','I','F','F',0,0,0,0,'W','A','V','E','f','m','t',' ',0x28,0x00,0x00,0x00,0xfe,0xff,0x06,0x00,0x44,0xac,0x00,0x00,0x30,0x13,0x08,0x00,0x0c,0x00,0x10,0x00,0x16,0x00,0x10,0x00,0x3f,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x10,0x00,0x80,0x00,0x00,0xaa,0x00,0x38,0x9b,0x71,'d','a','t','a',0,0,0,0};
    memset(buf, 0, sizeof(buf));

    printf("h2enc - converts 2 H2 ZOOM stereo waves into true surround 6ch wave. (c) 2011\nv1.4 by Simon [www.lehmayr.de]\n\n");
    if (argc < 2) {
        printf("insufficient parameters - front [and optionally rear] wav must be given!\n");
        return;
    }
    
    strcpy(foname, args[1]);
    s = strlen(foname)-1;
    while (foname[s] != '.' || s<=1) s--;
    foname[s] = 0; // dirty: cutting file extension
    strcat(foname, "-6ch.wav");

    ff = fopen(args[1], "rb");
    error(!ff, "cannot open front input file!");
    printf("front wave file:      %s\n", args[1]);
    offsf = determine_offset(ff, "data"); // looking for the one and only data chunk - ignoring data chunk size (just assuming that data lasts until end of file)
    fseek(ff, 0, SEEK_END);
    sizeff = ftell(ff)-offsf;

    if (argc == 3)
        strcpy(frname, args[2]);
    else {
        strcpy(frname, args[1]);
        s = strlen(frname)-1;
        while (frname[s] != '.' || s<=1) s--;
        if (frname[s-1] == 'F') frname[s-1] = 'R';
        else if (frname[s-1] == 'R') frname[s-1] = 'F';
        else if (frname[s-1] == 'f') frname[s-1] = 'r';
        else if (frname[s-1] == 'r') frname[s-1] = 'f';
        else error(1, "rear file not given and no auto-detect possible!");
    }
    fr = fopen(frname, "rb");
    error(!fr, "cannot open rear input file!");
    printf("rear wave file:       %s\n", frname);
    offsr = determine_offset(fr, "data");
    fseek(fr, 0, SEEK_END);
    sizefr = ftell(fr)-offsr;

    fo = fopen(foname, "wb");
    if (!fo) {printf("cannot open output file!\n"); return;}
    printf("6ch output wave file: %s\n", foname);
        
    hpos = determine_offset(ff, "fmt "); // looking for format header in src file
    fseek(ff, hpos, SEEK_SET);
    fread(&fmt_s, sizeof(fmt_s), 1, ff);
    samplerate = fmt_s.dwSamplesPerSec;
    samplewidth = fmt_s.wBitsPerSample;
    printf("sample rate (Hz):     %i\nsample width (bits):  %i\n", samplerate, samplewidth);
    fseek(ff, offsf, SEEK_SET);
    fseek(fr, offsr, SEEK_SET);
        
    if (sizeff == sizefr) { // only when input files match
        fosize = WAVEXSIZE+sizeff*3; // preparing output header
        hedr[4] = fosize;
        hedr[5] = fosize>>8;
        hedr[6] = fosize>>16;
        hedr[7] = fosize>>24;
        hedr[0x18] = samplerate;
        hedr[0x19] = samplerate>>8;
        blockalign = samplewidth*6/8;
        avgbytessec = blockalign*samplerate;
        hedr[0x1c] = avgbytessec;
        hedr[0x1d] = avgbytessec>>8;
        hedr[0x1e] = avgbytessec>>16;
        hedr[0x20] = blockalign;
        bitpersample = samplewidth;
        hedr[0x22] = bitpersample;
        hedr[0x26] = bitpersample;
        s = sizeff*3;
        hedr[64] = s;
        hedr[65] = s>>8;
        hedr[66] = s>>16;
        hedr[67] = s>>24;
        fwrite(hedr, 1, sizeof(hedr), fo); // storing header
        printf("writing %i bytes...\n\n0%%---------------------50%%--------------------100%%\n",fosize);
        curp = 0.0;
        while(s > 0) { // storing data
            fread(&buf[0], 1, samplewidth/4, ff); // building buffer
            fread(&buf[samplewidth/2], 1, samplewidth/4, fr);
            fwrite(buf, 1, blockalign, fo);
            s -= blockalign;
            realp = 100.0*(sizeff*3-s)/sizeff/3.0;
            if (realp > curp) {
                printf(">");
                curp += 2.0;
            }
        }
        printf("\ndone.\n");
    }
    else {
        printf("fatal error: quadrophonic file sizes not equal!\n");
    }
    fclose(ff);
    fclose(fr);
    fclose(fo);
}
