/*
Copyright (C) 2011 Simon Lehmayr <simon.lehmayr@gmx.de>

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>


/** standard error fn. if cond is true an error has occurred, so the text is printed ans program aborts
    @param cond - {[false, true]} true means error
    @param text - error message
*/
static void error(int cond, const char * text)
{
    if (cond) {
        printf("%s\n", text);
        exit(-1);
    }
}

/** general wave chunk eater. parses chunks in wave files.
    @param f - file to parse
    @param s - string name of current chunk
    @param readsize {[false, true]} - true means chunk size is to be read (and skipped)
    @param jump {[false, true]} - true means that chunk will be skipped
    @return start position of current chunk data (after chunk header)
*/
static unsigned long eat_chunk(FILE * f, char *s, int readsize, int jump)
{
    unsigned long size, pos;
    fread(s, 1, 4, f);
    if (readsize) fread(&size, 1, 4, f);
    pos = ftell(f);
    if (readsize && jump) fseek(f, size, SEEK_CUR);
    return pos;
}

/** header skipper. returns the start position of raw wave sample data.
    @param f - file to process
    @return start position of raw data
*/
static unsigned long determine_offset(FILE *f, char *name)
{
    char s[5];
    unsigned long pos;
    fseek(f, 0, SEEK_SET);
    eat_chunk(f, s, 1, 0);
    error(strncmp(s, "RIFF", 4) != 0, "not a wave file!");
    eat_chunk(f, s, 0, 0);
    error(strncmp(s, "WAVE", 4) != 0, "invalid wave file!");
    while(1) {
        pos = eat_chunk(f, s, 1, 1);
        error(feof(f), "invalid wave file!");
        if (strncmp(s, name, 4) == 0) break; 
    }
    return pos;
}


struct fmt_st {
    unsigned short wFormatTag;
    unsigned short wChannels;
    unsigned long dwSamplesPerSec;
    unsigned long dwAvgBytesPerSec;
    unsigned short wBlockAlign;
    unsigned short wBitsPerSample;
};

#define WAVEXFMT 0xFFFE
#define OFFSCHMASK 0x14

void main(int argc, char *args[])
{
    FILE *f;
    struct fmt_st fmt_s;
    unsigned long offsfmt;

    printf("patch_chmask - patches the channelmask in wavex files. (c) 2011\nv1.4 by Simon [www.lehmayr.de]\n\n");
    if (argc < 2) {
        printf("insufficient parameters - 6ch wave file must be given!\n");
        return;
    }
    
    f = fopen(args[1], "r+b");
    error(!f, "cannot open front input file!");
    printf("6ch wave file:      %s\n", args[1]);
    offsfmt = determine_offset(f, "fmt "); // looking for the one and only data chunk - ignoring data chunk size (just assuming that data lasts until end of file)
    fseek(f, offsfmt, SEEK_SET);
    fread(&fmt_s, sizeof(fmt_s), 1, f);
    error(fmt_s.wFormatTag != WAVEXFMT, "not a wavex file!");
    fseek(f, offsfmt+OFFSCHMASK, SEEK_SET);
    printf("old chmask = %.2X\n", fgetc(f));
    fseek(f, offsfmt+OFFSCHMASK, SEEK_SET);
    printf("set chmask to 0x3f\n");
    fputc(0x3f, f);
    printf("done.\n");
    fclose(f);
}
